#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <QNetworkAccessManager>
#include <QCryptographicHash>
#include <QJsonParseError>
#include <QDesktopWidget>
#include <QAuthenticator>
#include <QStandardPaths>
#include <QNetworkReply>
#include <QInputDialog>
#include <QMessageBox>
#include <QHeaderView>
#include <QSettings>
#include <QScreen>
#include <QStyle>
typedef std::pair<QNetworkRequest::Attribute, QString> PA;
static QList<PA> attributes = {PA(QNetworkRequest::HttpStatusCodeAttribute, "HttpStatusCodeAttribute")
    , PA(QNetworkRequest::HttpReasonPhraseAttribute, "HttpReasonPhraseAttribute")
    , PA(QNetworkRequest::RedirectionTargetAttribute, "RedirectionTargetAttribute")
    , PA(QNetworkRequest::ConnectionEncryptedAttribute, "ConnectionEncryptedAttribute")
    , PA(QNetworkRequest::CacheLoadControlAttribute, "CacheLoadControlAttribute")
    , PA(QNetworkRequest::CacheSaveControlAttribute, "CacheSaveControlAttribute")
    , PA(QNetworkRequest::SourceIsFromCacheAttribute, "SourceIsFromCacheAttribute")
    , PA(QNetworkRequest::DoNotBufferUploadDataAttribute, "DoNotBufferUploadDataAttribute")
    , PA(QNetworkRequest::HttpPipeliningAllowedAttribute, "HttpPipeliningAllowedAttribute")
    , PA(QNetworkRequest::HttpPipeliningWasUsedAttribute, "HttpPipeliningWasUsedAttribute")
    , PA(QNetworkRequest::CustomVerbAttribute, "CustomVerbAttribute")
    , PA(QNetworkRequest::CookieLoadControlAttribute, "CookieLoadControlAttribute")
    , PA(QNetworkRequest::CookieSaveControlAttribute, "CookieSaveControlAttribute")
    , PA(QNetworkRequest::AuthenticationReuseAttribute, "AuthenticationReuseAttribute")
    , PA(QNetworkRequest::BackgroundRequestAttribute, "BackgroundRequestAttribute")
    , PA(QNetworkRequest::SpdyAllowedAttribute, "SpdyAllowedAttribute")
    , PA(QNetworkRequest::SpdyWasUsedAttribute, "SpdyWasUsedAttribute")
    , PA(QNetworkRequest::HTTP2AllowedAttribute, "HTTP2AllowedAttribute")
    , PA(QNetworkRequest::HTTP2WasUsedAttribute, "HTTP2WasUsedAttribute")
    , PA(QNetworkRequest::EmitAllUploadProgressSignalsAttribute, "EmitAllUploadProgressSignalsAttribute")
    , PA(QNetworkRequest::FollowRedirectsAttribute, "FollowRedirectsAttribute")
    , PA(QNetworkRequest::OriginalContentLengthAttribute, "OriginalContentLengthAttribute")
    , PA(QNetworkRequest::RedirectPolicyAttribute, "RedirectPolicyAttribute")
    , PA(QNetworkRequest::Http2DirectAttribute, "Http2DirectAttribute")
};
QString windowName(const QString& text)
{
  return QObject::tr("%1 - %2")
      .arg(QCoreApplication::applicationName())
      .arg(text);
}

MainWindow::MainWindow(QWidget *parent) :
  QMainWindow(parent),
  ui(new Ui::MainWindow), networkAccessManager(new QNetworkAccessManager(this))
{
  ui->setupUi(this);
  setWindowTitle(QCoreApplication::applicationName());
  ui->action_Quit->setShortcut(QKeySequence::Quit);

  QSettings settings;
  if (!settings.contains("geometry"))
  {
    setGeometry(
          QStyle::alignedRect(
            Qt::LeftToRight,
            Qt::AlignCenter,
            size(),
            QGuiApplication::primaryScreen()->availableGeometry()
            )
          );
  }
  else
  {
    restoreGeometry(settings.value("geometry").toByteArray());
    restoreState(settings.value("windowState").toByteArray());
  }

  for(int i = 0, size = ui->headers->rowCount() ; i < size ; i++)
    ui->headers->setItem(i, 1, new QTableWidgetItem(settings.value(ui->headers->item(i, 0)->text(), "").toString()));


  connect(ui->url, SIGNAL(returnPressed()), ui->send, SLOT(click()));
  ui->headers->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
  connect(networkAccessManager, &QNetworkAccessManager::authenticationRequired, [&](QNetworkReply *, QAuthenticator *authenticator)
  {
    if (login.isNull())
      login = QStandardPaths::standardLocations(QStandardPaths::HomeLocation).at(0);

    login    = QInputDialog::getText(this, windowName("Authentification Required"),
                                     tr("User name:"), QLineEdit::Normal,login);
    password = QInputDialog::getText(this, windowName("Authentification Required"),
                                     tr("Password:"), QLineEdit::PasswordEchoOnEdit, password);
    authenticator->setUser(login);
    authenticator->setPassword(password);
  });
  connect(networkAccessManager, &QNetworkAccessManager::encrypted, [&](QNetworkReply *)
  {
    ui->statusBar->showMessage("Connection encrypted.", 2000);
  });
  connect(networkAccessManager, &QNetworkAccessManager::networkAccessibleChanged, [&](QNetworkAccessManager::NetworkAccessibility accessible)
  {
    if (accessible == QNetworkAccessManager::NotAccessible)
      ui->statusBar->showMessage("Network Not Accessible");
    else if(accessible == QNetworkAccessManager::UnknownAccessibility)
      ui->statusBar->showMessage("Network Status Unknown");
  });
  connect(networkAccessManager, &QNetworkAccessManager::preSharedKeyAuthenticationRequired, [&](QNetworkReply *, QSslPreSharedKeyAuthenticator *authenticator)
  {
    if (preSharedKeyLogin.isNull())
      preSharedKeyLogin = QStandardPaths::standardLocations(QStandardPaths::HomeLocation).at(0);
    preSharedKeyLogin    = QInputDialog::getText(this, windowName("PreSharedKey Auth. Required"),
                                                 tr("Identity:"), QLineEdit::Normal, preSharedKeyLogin);
    preSharedKeyPassword = QInputDialog::getText(this, windowName("PreSharedKey Auth. Required"),
                                                 tr("Key:"), QLineEdit::PasswordEchoOnEdit, preSharedKeyPassword);

    authenticator->setIdentity(qUtf8Printable(preSharedKeyLogin));
    authenticator->setPreSharedKey(qUtf8Printable(preSharedKeyPassword));
  });
  connect(networkAccessManager, &QNetworkAccessManager::proxyAuthenticationRequired, [&](const QNetworkProxy &, QAuthenticator *authenticator)
  {
    if (proxyLogin.isNull())
      proxyLogin = QStandardPaths::standardLocations(QStandardPaths::HomeLocation).at(0);
    proxyLogin    = QInputDialog::getText(this, windowName("Proxy Authentification Required"),
                                          tr("User name:"), QLineEdit::Normal, proxyLogin);
    proxyPassword = QInputDialog::getText(this, windowName("Proxy Authentification Required"),
                                          tr("Password:"), QLineEdit::PasswordEchoOnEdit, proxyPassword);

    authenticator->setUser(login);
    authenticator->setPassword(password);
  });
  connect(networkAccessManager, &QNetworkAccessManager::sslErrors, [&](QNetworkReply *reply, const QList<QSslError> &errors)
  {
    QMessageBox msgBox;
    msgBox.setWindowTitle(windowName("SSL Error"));
    msgBox.setText("SSL error encountered on connection.");
    msgBox.setInformativeText("You can have more information by clicking on the adhoc button\n"
                              "Do you want to ignore the warning?");
    msgBox.setDetailedText([](const QList<QSslError> &errors) -> QString
    {
      QString retour;
      for(auto error : errors)
        retour += error.errorString()+"\n";
      return retour;
    }(errors));
    msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    msgBox.setDefaultButton(QMessageBox::No);
    if (msgBox.exec() == QMessageBox::Yes)
      reply->ignoreSslErrors();
  });
  connect(networkAccessManager, &QNetworkAccessManager::finished, [&](QNetworkReply *reply)
  {
    if (reply->error() != QNetworkReply::NoError)
    {
      QMessageBox msgBox;
      msgBox.setWindowTitle(windowName("Error Encountered"));
      msgBox.setText("Error encountered on connection.");
      msgBox.setInformativeText("You can have more information by clicking on the adhoc button");
      msgBox.setDetailedText(reply->errorString());
      msgBox.exec();
    }
    ui->responseHeaders->clear();
    for(auto pair : reply->rawHeaderPairs())
      ui->responseHeaders->appendPlainText(QString("%1: %2")
                                           .arg(QString(pair.first))
                                           .arg(QString(pair.second)));
    ui->responseBody->setPlainText(reply->readAll());
    ui->responseAttributes->clear();
    for(auto pair : ::attributes)
      ui->responseAttributes->appendPlainText(QString("%1: %2")
                                              .arg(pair.second)
                                              .arg(reply->attribute(pair.first).toString()));

  });
}

MainWindow::~MainWindow()
{
  delete ui;
}

void MainWindow::on_responseBody_textChanged()
{
  QJsonParseError error;
  QJsonDocument doc = QJsonDocument::fromJson(qUtf8Printable(ui->responseBody->toPlainText()), &error);
  if (error.error != QJsonParseError::NoError)
    return;
  ui->responseBody->blockSignals(true);
  ui->responseBody->setPlainText(doc.toJson(QJsonDocument::Indented));
  ui->responseBody->blockSignals(false);
}
void MainWindow::closeEvent(QCloseEvent *event)
{
    QSettings settings;
    settings.setValue("geometry", saveGeometry());
    settings.setValue("windowState", saveState());
    for(int i = 0, size = ui->headers->rowCount() ; i < size ; i++)
      settings.setValue(ui->headers->item(i, 0)->text(), ui->headers->item(i, 1)->text());
    QMainWindow::closeEvent(event);
}
void MainWindow::on_body_textChanged()
{
  QJsonParseError error;
  ui->validity->clear();
  QJsonDocument doc = QJsonDocument::fromJson(qUtf8Printable(ui->body->toPlainText()), &error);
  if (error.error != QJsonParseError::NoError)
  {
    if ((QStringList() << "POST" << "PUT").contains(ui->method->currentText())
        || !ui->body->toPlainText().isEmpty()
        )
      ui->validity->setText(QString("Invalid JSON Body: %1").arg(error.errorString()));
  }
}
void MainWindow::on_send_clicked()
{
  QCryptographicHash hash(QCryptographicHash::Sha1);
  QNetworkRequest request(ui->url->text());
  for(int i = 0, size = ui->headers->rowCount() ; i < size ; i++)
    request.setRawHeader(qUtf8Printable(ui->headers->item(i, 0)->text())
                         , qUtf8Printable(ui->headers->item(i, 1)->text()));
  QString signatureHeaderValue;
  QString timestampHeaderValue;
  auto headerValue = [&](const QString& headerName) -> QString
  {
    for(int i = 0, size = ui->headers->rowCount() ; i < size ; i++)
      if (ui->headers->item(i, 0)->text() == headerName)
        return ui->headers->item(i, 1)->text();
    return "HEADER NOT FOUND";
  };
  auto updateHeader = [&](const QString& headerName, const QString& headerValue)
  {
    for(int i = 0, size = ui->headers->rowCount() ; i < size ; i++)
      if (ui->headers->item(i, 0)->text() == headerName)
        ui->headers->item(i, 1)->setText(headerValue);
  };
  auto updateHeaders = [&]()
  {
    updateHeader("X-Triethic-MTimestamp", timestampHeaderValue);
    updateHeader("X-Triethic-Signature", signatureHeaderValue);
  };
  timestampHeaderValue = QString::number(QDateTime::currentDateTime().toMSecsSinceEpoch());
  request.setRawHeader("X-Triethic-MTimestamp", qUtf8Printable(timestampHeaderValue));
  signatureHeaderValue = QString("%1+%2+%3")
               .arg(ui->applicationSecret->text())
               .arg(headerValue("X-Triethic-Consumer"))
               .arg(timestampHeaderValue);
  hash.addData(qUtf8Printable(signatureHeaderValue));
  signatureHeaderValue = "$1$"+hash.result().toHex();
  request.setRawHeader("X-Triethic-Signature", qUtf8Printable(signatureHeaderValue));

  switch(ui->method->currentIndex())
  {
    case 0://GET
      networkAccessManager->get(request);updateHeaders();
      break;
    case 1://POST
      networkAccessManager->post(request, qUtf8Printable(ui->body->toPlainText()));updateHeaders();
      break;
    case 2://PUT
      networkAccessManager->put(request, qUtf8Printable(ui->body->toPlainText()));updateHeaders();
      break;
    case 3://DELETE
      networkAccessManager->deleteResource(request);updateHeaders();
      break;
    default:
      QMessageBox::warning(this, windowName("Unsupported Method"), "Internal Error: this method should not be called");
      break;
  }
}
void MainWindow::on_action_Quit_triggered()
{
    close();
}
void MainWindow::on_actionQt_triggered()
{
   QMessageBox::about(this, "About Qt",
R"V0G0N(
Qt is the library used to this application, and this application was realized in less than a day, even though it uses JSON, Network (proxy, auth, ...), and so on.

Qt is a cross-platform application development framework for desktop, embedded and mobile. Supported Platforms include Linux, OS X, Windows, VxWorks, QNX, Android, iOS, BlackBerry, Sailfish OS and others.

Qt is available under various licenses: The Qt Company sells commercial licenses, but Qt is also available as free software under several versions of the GPL and the LGPL (see Licensing FAQ).
)V0G0N");
}
void MainWindow::on_actionApIOR_triggered()
{
  QMessageBox::about(this, "About ApIOR",
R"V0G0N(
Copyright 2018 Triethic By Gaia.

This small software was realized:
- to provide some Qt/C++ code showing how to use the API
- to demonstrate the functionnality of the API
- to provide a easy to use for developping and debugging

In short you can do anything you want with the code of
this software and we are liable for nothing.

                     Licensing.
The code is under the MIT License.
Except as contained in this notice, the name(s) of the
above copyright holders shall not be used in advertising
or otherwise to promote the sale, use or other dealings
in this Software or parts of it without prior written
authorization.


)V0G0N");
}
